-- SPDX-FileCopyrightText: 2023 The CC: Tweaked Developers
--
-- SPDX-License-Identifier: MPL-2.0

--[[- A parser for Lua programs and expressions.

> [!DANGER]
> This is an internal module and SHOULD NOT be used in your own code. It may
> be removed or changed at any time.

Most of the code in this module is automatically generated from the Lua grammar,
hence being mostly unreadable!

@local
]]

-- Lazily load our map of errors
local errors = setmetatable({}, {
    __index = function(self, key)
        setmetatable(self, nil)
        for k, v in pairs(require "cc.internal.syntax.errors") do self[k] = v end

        return self[key]
    end,
})

-- Everything below this line is auto-generated. DO NOT EDIT.

--- A lookup table of valid Lua tokens
local tokens = (function() return {} end)() -- Make tokens opaque to illuaminate. Nasty!
for i, token in ipairs({
    "WHILE", "UNTIL", "TRUE", "THEN", "SUB", "STRING", "SEMICOLON", "RETURN",
    "REPEAT", "POW", "OSQUARE", "OR", "OPAREN", "OBRACE", "NUMBER", "NOT",
    "NIL", "NE", "MUL", "MOD", "LT", "LOCAL", "LEN", "LE", "IN", "IF",
    "IDENT", "GT", "GOTO", "GE", "FUNCTION", "FOR", "FALSE", "EQUALS", "EQ",
    "EOF", "END", "ELSEIF", "ELSE", "DOUBLE_COLON", "DOTS", "DOT", "DO",
    "DIV", "CSQUARE", "CPAREN", "CONCAT", "COMMA", "COLON", "CBRACE",
    "BREAK", "AND", "ADD", "COMMENT", "ERROR",
}) do tokens[token] = i end
setmetatable(tokens, { __index = function(_, name) error("No such token " .. tostring(name), 2) end })

--- Read a integer with a given size from a string.
local function get_int(str, offset, size)
    if size == 1 then
        return str:byte(offset + 1)
    elseif size == 2 then
        local hi, lo = str:byte(offset + 1, offset + 2)
        return hi * 256 + lo
    elseif size == 3 then
        local b1, b2, b3 = str:byte(offset + 1, offset + 3)
        return b1 * 256 + b2 + b3 * 65536 -- Don't ask.
    else
        error("Unsupported size", 2)
    end
end

--[[ Error handling:

Errors are extracted from the current parse state in a two-stage process:
 - Run a DFA over the current state of the LR1 stack. For each accepting state,
   register a parse error.
 - Once all possible errors are found, pick the best of these and report it to
   the user.

This process is performed by a tiny register-based virtual machine. The bytecode
for this machine is stored in `error_program`, and the accompanying transition
table in `error_tbl`.

It would be more efficient to use tables here (`string.byte` is 2-3x slower than
a table lookup) or even define the DFA as a Lua program, however this approach
is much more space efficient - shaving off several kilobytes.

See https://github.com/let-def/lrgrep/ (namely ./support/lrgrep_runtime.ml) for
more information.
]]

local function is_same_line(context, previous, token)
    local prev_line = context.get_pos(previous)
    local tok_line = context.get_pos(token.s)
    return prev_line == tok_line and token.v ~= tokens.EOF
end

local function line_end_position(context, previous, token)
    if is_same_line(context, previous, token) then
        return token.s
    else
        return previous + 1
    end
end

local expr_tokens = {}
for _, v in pairs { tokens.STRING, tokens.NUMBER, tokens.TRUE, tokens.FALSE, tokens.NIL } do
  expr_tokens[v] = true
end

local error_messages = {
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 26
        if token.v == tokens.EQUALS then
        return errors.table_key_equals(token.s, token.e)
    end
    end,
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 34
        if token.v == tokens.EQUALS then
        return errors.use_double_equals(token.s, token.e)
    end
    end,
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 42
        if expr_tokens[token.v] then
      return errors.missing_table_comma(token.v, token.s, token.e, stack[stack_n + 2])
    end
    end,
    function(context, stack, stack_n, regs, token)
        local comma = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 52
        if token.v == tokens.CPAREN then
      return errors.trailing_call_comma(comma.s, comma.e, token.s, token.e)
    end
    end,
    function(context, stack, stack_n, regs, token)
        local lp = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 60
        return errors.unclosed_brackets(lp.s, lp.e, token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        local lp = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 62
        return errors.unclosed_brackets(lp.s, lp.e, token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        local lp = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 64
        return errors.unclosed_brackets(lp.s, lp.e, token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        local lp = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 66
        return errors.unclosed_label(lp.s, lp.e, token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        local loc = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 71
        if token.v == tokens.DOT then
        return errors.local_function_dot(loc.s, loc.e, token.s, token.e)
    end
    end,
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 79
        local end_pos = stack[stack_n + 2] -- Hack to get the last position
    if is_same_line(context, end_pos, token) then
        return errors.standalone_name(token.s)
    else
        return errors.standalone_name_call(end_pos)
    end
    end,
    function(context, stack, stack_n, regs, token)
        local start = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 90
        return errors.expected_then(start.s, start.e, line_end_position(context, stack[stack_n + 2], token))
    end,
    function(context, stack, stack_n, regs, token)
        local start = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 118
        return errors.expected_end(start.s, start.e, token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        local func = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        local loc = { s = stack[regs[3] + 1], e = stack[regs[3] + 2] }
        -- parse_errors.mlyl, line 122
        return errors.expected_end(loc.s, func.e, token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 126
        if token.v == tokens.END then
      return errors.unexpected_end(token.s, token.e)
    elseif token ~= tokens.EOF then
        return errors.expected_statement(token.v, token.s, token.e)
    end
    end,
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 136
        return errors.expected_function_args(token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 140
        return errors.expected_expression(token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 144
        return errors.expected_var(token.v, token.s, token.e)
    end,
}
local error_program_start, error_program = 491, "\6\1\0\3\0072\0\3\7\186\0\3\7\168\0\3\7\160\0\3\7\152\0\3\7\141\0\3\7\127\0\3\7u\0\3\7d\0\3\7W\0\1\0\3\7C\0\3\7;\0\1\0\3\0072\0\3\7*\0\3\7\"\0\3\7\"\0\3\7\"\0\3\7\"\0\3\7\"\0\3\7\n\0\3\7\n\0\3\7\"\0\3\7\26\0\3\7\18\0\3\7\18\0\3\7\18\0\3\7\n\0\3\7\2\0\3\6\232\0\3\6\224\0\3\6\216\0\3\6\208\0\3\6\200\0\3\6\22\0\3\6\190\0\3\6\186\0\3\6\175\0\3\6#\0\3\6\31\0\3\6\23\0\3\1\246\0\3\6\22\0\3\6\16\0\3\6\n\0\3\6\4\0\3\5\255\0\3\5\245\0\3\5\237\0\3\5\226\0\3\5\193\0\3\5\188\0\1\0\3\5\183\0\1\0\3\5\178\0\3\5\165\0\3\5\154\0\3\4\235\0\3\2\29\0\3\0048\0\3\3\155\0\3\3\147\0\3\3\143\0\3\3\135\0\3\3w\0\3\3\139\0\3\3\135\0\3\3\135\0\3\3\131\0\3\3\127\0\3\2\249\0\3\3{\0\3\2\234\0\3\2\234\0\3\3w\0\3\3p\0\3\2\234\0\3\2\249\0\3\2\234\0\3\2\234\0\3\2\234\0\3\2\234\0\3\2\241\0\3\2\234\0\3\2\234\0\3\2\234\0\3\2\234\0\3\2a\0\3\2)\0\3\2U\0\3\2Y\0\3\2)\0\3\2U\0\3\2U\0\3\2Q\0\3\2M\0\3\2I\0\3\2%\0\3\2=\0\3\2E\0\3\0029\0\3\2A\0\3\2=\0\3\0029\0\3\0025\0\3\0021\0\3\2-\0\3\2)\0\3\2%\0\3\2!\0\3\2\29\0\3\2\25\0\3\2\21\0\3\2\21\0\3\2\17\0\3\2\r\0\3\2\t\0\3\2\t\0\3\1\255\0\3\2\5\0\3\1\255\0\3\1\246\0\5\0\0\3\5\250\0\3\7\197\0\5\0\15\1\0\3\5\178\0\1\0\3\5\178\0\3\7\197\0\3\7\205\0\3\7\205\0\3\2\r\0\3\7\214\0\3\7\218\0\3\5\188\0\3\7\222\0\3\2A\0\3\8\2\0\3\5\226\0\3\2-\0\3\0021\0\3\2%\0\3\2)\0\3\2U\0\3\2Q\0\3\2E\0\3\8\t\0\3\0029\0\3\2=\0\3\8\2\0\3\2-\0\5\0\5\1\0\3\5\178\0\1\0\3\8\149\0\3\8\133\0\1\0\3\8|\0\1\0\3\8s\0\1\0\3\8j\0\3\8b\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8B\0\3\8B\0\3\8Z\0\3\8R\0\3\8J\0\3\8J\0\3\8J\0\3\8B\0\3\8:\0\3\0082\0\3\8*\0\3\8*\0\3\8\"\0\3\8\26\0\3\8\18\0\3\2\234\0\3\2\234\0\3\2\234\0\3\3\135\0\5\0\196\3\8\r\0\3\3w\0\3\8\166\0\5\0\8\3\t.\0\1\0\3\8\149\0\3\8\133\0\1\0\3\8|\0\1\0\3\8s\0\1\0\3\8j\0\3\8b\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8B\0\3\8B\0\3\8Z\0\3\8R\0\3\8J\0\3\8J\0\3\8J\0\3\8B\0\3\8:\0\3\0082\0\3\8*\0\3\8*\0\3\8\"\0\3\8\26\0\3\8\18\0\5\1\n\3\8\r\0\3\3\135\0\3\t5\0\3\8\166\0\3\t.\0\3\2\234\0\3\t9\0\3\3\147\0\3\t=\0\3\8\133\0\5\0\3\1\0\3\8\149\0\3\tK\0\1\0\3\8|\0\1\0\3\8s\0\1\0\3\8j\0\3\5\188\0\1\0\3\5\178\0\3\8b\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8B\0\3\8B\0\3\8Z\0\3\8R\0\3\8J\0\3\8J\0\3\8J\0\3\8B\0\3\8:\0\3\0082\0\3\8*\0\3\8*\0\3\8\"\0\3\8\26\0\3\8\18\0\3\2\234\0\3\2\234\0\3\2\234\0\3\3\135\0\3\tA\0\3\2)\0\3\7\218\0\3\7\197\0\5\1p\3\8\r\0\1\0\3\8\149\0\3\8\133\0\3\tK\0\1\0\3\8|\0\1\0\3\8s\0\1\0\3\8j\0\3\5\188\0\1\0\3\tS\0\1\0\3\5\178\0\3\8b\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8B\0\3\8B\0\3\8Z\0\3\8R\0\3\8J\0\3\8J\0\3\8J\0\3\8B\0\3\8:\0\3\0082\0\3\8*\0\3\8*\0\3\8\"\0\3\8\26\0\3\8\18\0\3\t=\0\3\2\234\0\3\2\234\0\3\2\234\0\3\3\135\0\3\2)\0\3\tA\0\3\2)\0\3\7\218\0\3\7\197\0\5\1\243\3\8\r\0\1\0\3\8\149\0\3\8\133\0\3\tK\0\1\0\3\8|\0\1\0\3\8s\0\1\0\3\8j\0\1\0\3\tS\0\1\0\3\5\178\0\3\8b\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8B\0\3\8B\0\3\8Z\0\3\8R\0\3\8J\0\3\8J\0\3\8J\0\3\8B\0\3\8:\0\3\0082\0\3\8*\0\3\8*\0\3\8\"\0\3\8\26\0\3\8\18\0\3\t=\0\3\2\234\0\3\2\234\0\3\2\234\0\3\3\135\0\3\2)\0\3\tA\0\3\2)\0\3\7\218\0\3\7\197\0\5\2\133\3\8\r\0\3\2\r\0\4\2\0\0\5\0\31\1\0\3\5\183\0\4\4\0\1\6\4\5\0\1\6\4\t\0\0\6\4\r\0\0\6\1\0\3\tX\0\3\tp\0\3\tl\0\3\th\0\3\ta\0\1\0\3\tX\0\4\r\0\0\5\1\28\3\t~\0\4\r\0\0\3\t\161\0\4\14\0\0\6\4\15\0\0\6\4\16\0\0\6\1\0\3\5\178\0\1\0\3\5\183\0\1\0\3\tS\0\6\3\tK\0\3\2)\0\5\0\12\6\3\tA\0\1\0\3\8\149\0\3\8\133\0\1\0\3\8|\0\1\0\3\8s\0\1\0\3\8j\0\3\5\188\0\3\8b\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8B\0\3\8B\0\3\8Z\0\3\8R\0\3\8J\0\3\8J\0\3\8J\0\3\8B\0\3\8:\0\3\0082\0\3\8*\0\3\8*\0\3\8\"\0\3\8\26\0\3\8\18\0\3\2\234\0\3\2\234\0\3\2\234\0\3\3\135\0\3\7\218\0\5\3\23\3\8\r\0\3\th\0\5\0 \6\3\tl\0\1\0\3\t\168\0\4\1\0\0\5\0E\6\4\1\0\0\3\t\173\0\4\1\0\0\3\t\177\0\4\1\0\0\3\2U\0\4\1\0\0\3\t\181\0\1\0\3\5\178\0\3\2=\0\3\2U\0\3\2-\0\4\1\0\0\5\0\143\6\4\1\0\0\3\t\185\0\4\1\0\0\3\t\189\0\4\1\0\0\3\t\193\0\4\1\0\0\3\t\197\0\4\1\0\0\3\t\201\0\4\r\0\0\4\11\0\1\6\4\14\0\0\3\t\205\0\4\15\0\0\3\t\215\0\3\t~\0\3\tp\0\3\ta\0\5\0\131\1\0\3\tX\0\3\t\219\0\4\1\0\0\4\0\0\0\5\0)\1\0\3\t\227\0\4\1\0\0\1\0\3\5\178\0\4\1\0\0\1\0\3\t\236\0\3\tp\0\4\r\0\0\5\0\12\3\t\241\0\4\r\0\0\3\tp\0\4\r\0\0\3\t\247\0\4\r\0\0\1\0\3\tX\0\3\tp\0\3\t\253\0\4\r\0\0\5\0\171\3\n\1\0\3\1\255\0\3\2\r\0\5\0Z\1\0\3\5\183\0\3\n\5\0\3\2\29\0\3\n\19\0\1\0\3\0072\0\3\7\168\0\3\7\160\0\3\7\152\0\3\7\141\0\1\0\3\0072\0\3\5\193\0\5\1P\3\7\186\0\3\2I\0\4\1\0\0\6\4\1\0\0\3\n\26\0\4\1\0\0\3\n\"\0\4\1\0\0\3\5\226\0\4\1\0\0\3\2=\0\4\1\0\0\3\2)\0\4\1\0\0\3\n=\0\4\1\0\0\3\n\157\0\4\1\0\0\3\11\8\0\4\1\0\0\3\11G\0\4\1\0\0\3\11\142\0\4\1\0\0\3\11\209\0\4\4\0\1\4\1\0\0\6\4\6\0\1\4\1\0\0\6\4\n\0\1\4\1\0\0\6\4\2\0\0\4\1\0\0\4\0\0\0\3\7\205\0\4\5\0\1\4\2\0\0\4\1\0\0\4\0\0\0\6\3\3\131\0\1\0\3\8\149\0\3\8\133\0\1\0\3\8|\0\1\0\3\8s\0\1\0\3\8j\0\3\8b\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8B\0\3\8B\0\3\8Z\0\3\8R\0\3\8J\0\3\8J\0\3\8J\0\3\8B\0\3\8:\0\3\0082\0\3\8*\0\3\8*\0\3\8\"\0\3\8\26\0\3\8\18\0\3\2\234\0\3\2\234\0\3\2\234\0\3\3\135\0\3\8\2\0\5\3_\3\8\r\0\3\2\249\0\3\12`\0\3\6\175\0\3\6\31\0\1\0\3\12g\0\4\14\0\0\5\0P\6\4\7\0\1\6\4\11\0\1\6\3\tp\0\5\0006\3\t\241\0\3\tp\0\3\t\247\0\1\0\3\tX\0\3\tp\0\3\t\253\0\5\0\173\3\n\1\0\1\0\3\tX\0\3\tp\0\3\tl\0\3\th\0\3\ta\0\1\0\3\tX\0\5\1\199\3\t~\0\4\n\0\1\6\3\n\26\0\3\n\"\0\3\n=\0\3\n\157\0\3\11\8\0\3\11G\0\3\11\142\0\3\11\209\0\1\0\3\12g\0\3\12l\0\5\0\203\6\4\2\0\0\3\7\205\0\4\5\0\1\4\2\0\0\6\4\6\0\1\6\1\0\3\12s\0\1\0\3\12|\0\3\12\132\0\3\12\136\0\3\5\245\0\1\0\3\0072\0\3\12\140\0\5\0\196\3\7\168\0\3\t\177\0\3\t\219\0\5\0p\1\0\3\t\227\0\1\0\3\5\178\0\3\2=\0\3\2U\0\3\2-\0\5\0\145\6\1\0\3\8\149\0\3\8\133\0\1\0\3\8|\0\1\0\3\8s\0\1\0\3\8j\0\3\8b\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8B\0\3\8Z\0\3\8R\0\3\8:\0\3\0082\0\3\8*\0\3\8*\0\3\8\"\0\3\8\26\0\3\8\18\0\5\3\165\3\8\r\0\1\0\3\8\149\0\3\8\133\0\1\0\3\8|\0\1\0\3\8s\0\1\0\3\8j\0\3\8b\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8B\0\3\8B\0\3\8Z\0\3\8R\0\3\8B\0\3\8:\0\3\0082\0\3\8*\0\3\8*\0\3\8\"\0\3\8\26\0\3\8\18\0\5\3\235\3\8\r\0\1\0\3\8\149\0\3\8\133\0\1\0\3\8|\0\1\0\3\8s\0\1\0\3\8j\0\3\8:\0\3\0082\0\3\8*\0\3\8*\0\3\8\"\0\3\8\26\0\3\8\18\0\5\1\254\3\8\r\0\1\0\3\8\149\0\3\8\133\0\1\0\3\8|\0\1\0\3\8s\0\1\0\3\8j\0\3\8b\0\3\8R\0\3\8:\0\3\0082\0\3\8*\0\3\8*\0\3\8\"\0\3\8\26\0\3\8\18\0\5\3\242\3\8\r\0\1\0\3\8\149\0\3\8\133\0\1\0\3\8|\0\1\0\3\8s\0\1\0\3\8j\0\3\8R\0\3\8:\0\3\0082\0\3\8*\0\3\8*\0\3\8\"\0\3\8\26\0\3\8\18\0\5\0048\3\8\r\0\1\0\3\8\149\0\3\8\133\0\1\0\3\8|\0\1\0\3\8s\0\1\0\3\8j\0\3\8b\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8Z\0\3\8B\0\3\8B\0\3\8Z\0\3\8R\0\3\8J\0\3\8J\0\3\8J\0\3\8B\0\3\8:\0\3\0082\0\3\8*\0\3\8*\0\3\8\"\0\3\8\26\0\3\8\18\0\3\2\234\0\3\2\234\0\3\2\234\0\3\t.\0\3\8\166\0\3\3\135\0\5\4\171\3\8\r\0\4\8\0\1\6\5\0\128\3\12\150\0\2\1\0\1\0\3\12\155\0\4\11\0\1\3\6\22\0\3\tp\0\3\12\164\0\4\r\0\0\1\0\3\12s\0\4\3\0\1\6\4\12\1\2\6\3\12\171\0\5\0\247\3\12\175\0\3\tp\0\3\12\179\0\3\12\171\0"
local error_tbl_ks, error_tbl_vs, error_tbl = 1, 2, "\0\0\203\1\0\147\2\0\0\3\0\199\0\0\0\5\1[\0\0\0\7\1W\0\0\0\t\0\211\0\0\0\11\1S\0\0\0\r\1O\0\0\0\15\0\231\16\0\227\17\0\191\18\0\217\19\1\231\20\0\167\21\1\227\22\0\0\23\0\239\24\1\223\25\0\183\26\1\183\27\1\219\28\0\11\29\0'\30\1K\31\1G \1C!\0\151\"\1?#\1;$\0017\22\1\242&\0013'\1/\0\0\0)\1+*\1'+\1#,\0\27\0\0\0.\0o\0\0\0000\0k\0\0\0002\0g\0\0\0004\0c\0\0\0006\0_\0\0\0008\0[\0\0\0:\0W\0\0\0<\0S\0\0\0>\0O\0\0\0@\0K\0\0\0B\0G\0\0\0D\0C\0\0\0F\0?\0\0\0H\0;I\0\251J\0\217K\1_L\0+M\0wN\0\183O\1\31P\0sQ\0\0R\0\247S\0\0T\0\0U\1\27V\1\23W\1\19X\1\15Y\1\11Z\0\27[\0\0T\2\245]\0\131^\0\223_\1\215`\1\211a\1\207b\1\203c\0\243a\3\151e\0\127f\0#g\0\179h\1\7i\0\31j\1\3k\0\255l\0\135m\0\7n\0'o\1\179p\1\175q\1\171r\1\167s\1\163t\0\203u\0\147q\2]w\0{x\0\0y\0\187z\0001{\0\23|\0'}\1\159~\0\159\127\1\155a\5\161\129\1\151\130\0\187\131\0\155x\6\27\133\0\135\134\0\19\135\0\147\136\0\139\137\1\147a\7`\139\0\135\140\0\15\141\0\143\142\0005\143\0'\144\1\143\145\0\235\146\1\139\147\0\187\148\0\163\149\0\191\150\0\0\151\1\199\152\0\0\153\1\195\154\0\23\155\0'\156\1\135\157\0\187\158\0\171\28\7O\160\0\135\149\7\137\162\0\135\163\0\7\164\0'\165\1\131\134\6\182\167\0\135\168\0\7\169\0'\170\1\127\171\0\0\140\6\182\173\0\0\174\0\7\175\0'\176\1{\177\0\0\178\0\175\179\1w\180\0\1\181\0'\182\1s\183\1o\184\1k\185\0\0\186\0\207a\7\201\188\1g\189\0\187\190\1\191\191\1\187\192\0\195\193\1c\194\0\0\195\0\0\196\0\0\197\0\0\198\0\0\3\0\0x\tE\132\6\194\6\2\218\149\t]\8\2\128\t\2j\n\2z\138\6\194\12\2\222a\n\30\14\2\226\8\t\211T\0\0\17\n\t\0\0\0\0\0\0\0\0\0J\6\240\0\0\0J\n+\0\0\0\18\t\211" .. ("\0"):rep(33) .. "%\2z\0\0\0\0\0\0(\2\230\0\0\0\0\0\0\0\0\0m\7K-\2\186\0\0\0/\2\182\0\0\0001\2\178K\7\1823\2\174K\tz5\2\170\0\0\0007\2\166\0\0\0009\2\162{\7S;\2\158q\6\254=\2\154q\n9?\2\150\0\0\0A\2\146\0\0\0C\2\142\0\0\0E\2\138\0\0\0G\2\134\0\0\0\0\0\0J\2\128\128\6\250L\2\190\128\n5\8\3\22\t\3\0\n\3\16J\t\211\0\0\0l\7\178\0\0\0l\tv\0\0\0\0\0\0\0\0\0\154\7S\0\0\0\0\0\0\\\2\214\0\0\0\0\0\0\0\0\0\0\0\0a\2p\163\7K\0\0\0d\2\210\0\0\0\0\0\0\168\7K\129\7\182\0\0\0\129\tz%\3\16\0\0\0\174\7K\0\0\0\0\0\0\0\0\0q\2\206\0\0\0-\3P\28\5\204/\3Lv\2\2021\3H\0\0\0003\3Dz\n\0155\3@\0\0\0007\3<\0\0\0009\0038\128\2\198;\0034\0\0\0=\0030\132\2t?\3,\187\6\246A\3(\187\n1C\3$\138\2tE\3 \0\7\254G\3\28\0\0\0\\\12\160J\3\22\0\0\0L\3T\0\0\0\173\7\182\0\0\0\173\tzd\12\160" .. ("\0"):rep(30) .. "\\\3l\188\7\182\0\0\0\188\tz\0\0\0a\3\6\28\7\232\0\0\0d\3h\196\7\182\0\0\0\196\tz\0\0\0\3\3\186\0\0\0\0\0\0\6\4\24\128\12\160\8\3\180\0\0\0\n\3\174q\3d\12\4\28\0\0\0\14\4 \187\2\194v\3`\189\0\0\18\3\190\0\0\0\0\0\0\0\0\0\22\0044\0\0\0\0\0\0\0\0\0\128\3\\\0\0\0\0\0\0\0\0\0\132\3\n\0\0\0t\0\0\0\0\0\0\0\0\0\0\0\138\3\n%\3\174\159\12\160{\5\216(\4$\0\0\0\0\0\0\0\0\0\0\0\0-\3\248\0\0\0/\3\244\0\0\0001\3\240\134\5\2123\3\236\0\0\0005\3\232\0\0\0007\3\228\140\5\2089\3\224\142\5\220;\3\220\0\0\0=\3\216\0\0\0?\3\212\0\0\0A\3\208\187\12\160C\3\204\0\0\0E\3\200\154\5\216G\3\196\0\0\0\0\0\0J\3\180\0\0\0L\3\252" .. ("\0"):rep(21) .. "t\7\254\187\3X" .. ("\0"):rep(15) .. "{\7\244\\\4\20\0\0\0\0\0\0\0\0\0\180\5\198\0\0\0\0\0\0\0\0\0d\4\16\0\0\0\134\7\240" .. ("\0"):rep(15) .. "\140\7\236\0\0\0\142\7\248\0\0\0\0\0\0q\4\12\0\0\0\28\t\139\0\0\0\0\0\0v\4\8\0\0\0x\4,y\3\164\154\7\244" .. ("\0"):rep(15) .. "\128\4\4\0\0\0\130\4(\0\0\0\132\3\168\0\0\0\3\4_\0\0\0\0\0\0\6\4\199\138\3\168\8\4Y\t\4?\n\4S\0\0\0\12\4\203\0\0\0\14\4\207\0\0\0\147\3\164\180\7\226\18\4i\8\11%\t\11\15\n\11\31\22\4\231\0\0\0\0\0\0\0\0\0\157" .. ("\0"):rep(32) .. "%\4S\0\0\0\0\0\0(\4\211\0\0\0\0\0\0\0\0\0\0\0\0-\4\163\0\0\0/\4\159%\11\0311\4\155\0\0\0003\4\151\0\0\0005\4\147\0\0\0007\4\143\187\4\0009\4\139\189\0040;\4\135\0\0\0=\4\131\0\0\0?\4\127\0\0\0A\4{\0\0\0C\4w\0\0\0E\4s\0\0\0G\4ot\0\0\0\0\0J\4Y\0\0\0L\4\167\0\0\0\0\0\0{\t\151\0\0\0Q\4\195\0\0\0S\0\0\0\0\0J\11%\0\0\0L\11+\0\0\0\0\0\0\134\t\147\0\0\0\\\4\191\0\0\0\0\0\0\0\0\0\140\t\143a\4E\142\t\155\0\0\0d\4\187\0\0\0\0\0\0\\\11C\0\0\0\0\0\0\0\0\0\0\0\0a\11\21\0\0\0\154\t\151d\11?\0\0\0q\4\183\0\0\0\0\0\0\0\0\0\0\0\0v\4\179\0\0\0x\4\223y\4I\0\0\0\0\0\0q\11;\0\0\0\0\0\0\0\0\0\128\4\175v\0117\130\4\219\0\0\0\132\4M\0\0\0\0\0\0\0\0\0\180\t\133\0\0\0\138\4M\128\0113\0\0\0\0\0\0\0\0\0\132\11\25\0\0\0\145\4\215\0\0\0\147\4I\0\0\0\138\11\25\0\0\0\0\0\0\6\5v\0\0\0\8\5\12\t\4\242\n\5\6\157\0\0\12\5z\0\0\0\14\5~\0\0\0\0\0\0\0\0\0\18\5\24\0\0\0\0\0\0\0\0\0\22\5\150" .. ("\0"):rep(24) .. "\177\4c" .. ("\0"):rep(15) .. "%\5\6\0\0\0\0\0\0(\5\130\187\4\171\0\0\0\189\4\227\0\0\0-\5R\0\0\0/\5N\0\0\0001\5J\0\0\0003\5F\187\11/5\5B\0\0\0007\5>\0\0\0009\5:\0\0\0;\0056\0\0\0=\0052\0\0\0?\5.\0\0\0A\5*\0\0\0C\5&\0\0\0E\5\"\0\0\0G\5\30\0\0\0\0\0\0J\5\12\0\0\0L\5V\0\0\0\0\0\0\0\0\0\0\0\0Q\5r\0\0\0S" .. ("\0"):rep(26) .. "\\\5n\0\0\0\0\0\0\0\0\0\0\0\0a\4\248\0\0\0\0\0\0d\5j" .. ("\0"):rep(36) .. "q\5f\0\0\0\0\0\0\0\0\0\0\0\0v\5b\0\0\0x\5\142y\4\252" .. ("\0"):rep(18) .. "\128\5^\0\0\0\130\5\138\0\0\0\132\5" .. ("\0"):rep(16) .. "\138\5" .. ("\0"):rep(19) .. "\145\5\134\0\0\0\147\4\252\0\0\0\3\6C\0\0\0\0\0\0\6\6\155\0\0\0\8\6=\t\6'\n\0067\157\0\0\12\6\159\0\0\0\14\6\163" .. ("\0"):rep(48) .. "\177\5\18" .. ("\0"):rep(15) .. "%\0067\0\0\0\0\0\0(\6\167\187\5Z\0\0\0\189\5\146\0\0\0-\6{\0\0\0/\6w\0\0\0001\6s\0\0\0003\6o\0\0\0005\6k\0\0\0007\6g\0\0\0009\6c\0\0\0;\6_\0\0\0=\6[\0\0\0?\6W\0\0\0A\6S\0\0\0C\6O\0\0\0E\6K\0\0\0G\6G\0\0\0\0\0\0J\6=\3\t*L\6\127\0\0\0\6\t\26\0\0\0\8\8\192\t\8\170\n\8\186\0\0\0\12\t\30\0\0\0\14\t\"" .. ("\0"):rep(15) .. "\\\6\151\0\0\0\0\0\0\0\0\0\0\0\0a\6-\0\0\0\0\0\0d\6\147" .. ("\0"):rep(24) .. "%\8\186\0\0\0\0\0\0(\t&q\6\143\0\0\0\0\0\0\0\0\0-\8\250v\6\139/\8\246\0\0\0001\8\242\0\0\0003\8\238\0\0\0005\8\234\0\0\0007\8\230\128\6\1359\8\226\0\0\0;\8\222\132\0061=\8\218\0\0\0?\8\214\0\0\0A\8\210\138\0061C\8\206\0\0\0E\8\202\0\0\0G\8\198\0\0\0\0\0\0J\8\192\0\0\0L\8\254\0\0\0\8\nW\t\nA\n\nQ" .. ("\0"):rep(33) .. "\\\t\22\0\0\0\0\0\0\0\0\0\0\0\0a\8\176\0\0\0\0\0\0d\t\18" .. ("\0"):rep(18) .. "%\nQ" .. ("\0"):rep(15) .. "q\t\14\0\0\0\187\6\131\0\0\0\189\6\171v\t\n\0\0\0\0\0\0\0\0\0\0\0\0005\n}\0\0\0007\ny\0\0\0009\nu\128\t\6\0\0\0\0\0\0=\nq\132\8\180?\nm\0\0\0A\ni\0\0\0C\ne\138\8\180E\na\0\0\0G\n]\0\0\0\0\0\0J\nW\0\0\0L\n\129\0\0\0\8\n\186\t\n\164\n\n\180\0\0\0\0\0\0\0\0\0\0\0\0\8\11d\t\11N\n\11^\0\0\0\0\0\0\0\0\0\0\0\0\\\n\153\0\0\0\0\0\0\0\0\0\0\0\0a\nG\0\0\0\0\0\0d\n\149" .. ("\0"):rep(18) .. "%\n\180" .. ("\0"):rep(15) .. "q\n\145%\11^-\n\232\0\0\0\187\t\2v\n\141\189\0\0\0\0\0\0\0\0\0\0\0005\n\228\0\0\0007\n\224\0\0\0009\n\220\128\n\137;\n\2165\11n=\n\212\132\nK?\n\208\0\0\0A\n\204\0\0\0C\n\200\138\nKE\n\196\0\0\0G\n\192\0\0\0\0\0\0J\n\186\0\0\0L\n\236\0\0\0G\11j\0\0\0\0\0\0J\11d\0\0\0L\11r\0\0\0\8\11\171\t\11\149\n\11\165\0\0\0\0\0\0\0\0\0\0\0\0\\\11\4\0\0\0\0\0\0\0\0\0\0\0\0a\n\170\0\0\0\\\11\138d\11\0\0\0\0\0\0\0\0\0\0a\11T\0\0\0\0\0\0d\11\134" .. ("\0"):rep(15) .. "q\n\252%\11\165\0\0\0\0\0\0\187\n\133v\n\248\0\0\0q\11\130\0\0\0\0\0\0\0\0\0\0\0\0v\11~\0\0\0\0\0\0\128\n\244\0\0\0005\11\177\0\0\0\132\n\174\0\0\0\0\0\0\128\11z\0\0\0\0\0\0\138\n\174\132\11X" .. ("\0"):rep(15) .. "\138\11X" .. ("\0"):rep(15) .. "J\11\171\0\0\0L\11\181" .. ("\0"):rep(45) .. "\\\11\205\0\0\0\0\0\0\0\0\0\0\0\0a\11\155\0\0\0\0\0\0d\11\201" .. ("\0"):rep(27) .. "\187\n\240\0\0\0\0\0\0q\11\197\0\0\0\0\0\0\0\0\0\187\11vv\11\193\0\0\0\0\0\0\6\12H\0\0\0\8\11\238\t\11\216\n\11\232\0\0\0\12\12L\128\11\189\14\12P\0\0\0\0\0\0\132\11\159" .. ("\0"):rep(15) .. "\138\11\159" .. ("\0"):rep(33) .. "#\12T\0\0\0%\11\232\0\0\0\0\0\0(\12\\\0\0\0\0\0\0\0\0\0\0\0\0-\12(\0\0\0/\12$\0\0\0001\12 \0\0\0003\12\28\0\0\0005\12\24\0\0\0007\12\20\0\0\0009\12\16\0\0\0;\12\12\0\0\0=\12\8\0\0\0?\12\4\0\0\0A\12\0\0\0\0C\11\252\0\0\0E\11\248\0\0\0G\11\244\187\11\185\0\0\0J\11\238\0\0\0L\12," .. ("\0"):rep(21) .. "T\12X" .. ("\0"):rep(21) .. "\\\12D\0\0\0\0\0\0\0\0\0\0\0\0a\11\222\0\0\0\0\0\0d\12@" .. ("\0"):rep(36) .. "q\12<\0\0\0\0\0\0\0\0\0\0\0\0v\0128" .. ("\0"):rep(27) .. "\128\0124\0\0\0\0\0\0\0\0\0\132\11\226" .. ("\0"):rep(15) .. "\138\11\226" .. ("\0"):rep(138) .. "\185\12T\0\0\0\187\0120"

local function handle_error(context, stack, stack_n, token, token_start, token_end)
    -- Run our error handling virtual machine.
    local pc, top, registers, messages = error_program_start, stack_n, {}, {}
    while true do
        local instruction = error_program:byte(pc + 1)
        if instruction == 1 then -- Store
            registers[error_program:byte(pc + 2) + 1] = top
            pc = pc + 2
        elseif instruction == 2 then -- Move
            registers[error_program:byte(pc + 2) + 1] = registers[error_program:byte(pc + 3) + 1]
            pc = pc + 3
        elseif instruction == 3 then -- Pop one item from the stack and jump
            if top > 1 then top = top - 3 end
            pc = get_int(error_program, pc + 1, 3)
        elseif instruction == 4 then -- Accept
            local clause, _, count = error_program:byte(pc + 2, pc + 4)
            local accept = { clause + 1 }
            for i = 1, count do accept[i + 1] = registers[count - i + 1] end
            messages[#messages + 1] = accept

            pc = pc + 4
        elseif instruction == 5 then -- Match
            local hi, lo = error_program:byte(pc + 2, pc + 3)
            local lr1 = stack[top] - 1

            local offset = (hi * 256 + lo + lr1) * (error_tbl_ks + error_tbl_vs)
            if offset + error_tbl_ks + error_tbl_vs <= #error_tbl and
                get_int(error_tbl, offset, error_tbl_ks) == lr1 then
                pc = get_int(error_tbl, offset + error_tbl_ks, error_tbl_vs)
            else
                pc = pc + 3
            end
        elseif instruction == 6 then -- Halt
            break
        else
            error("Illegal instruction while handling errors " .. tostring(instruction))
        end
    end

    -- Sort the list to ensure earlier patterns are used first.
    table.sort(messages, function(a, b) return a[1] < b[1] end)

    -- Then loop until we find an error message which actually works!
    local t = { v = token, s = token_start, e = token_end }
    for i = 1, #messages do
        local action = messages[i]
        local message = error_messages[action[1]](context, stack, stack_n, action, t)
        if message then
            context.report(message)
            return false
        end
    end

    context.report(errors.unexpected_token, token, token_start, token_end)
    return false
end

--- The list of productions in our grammar. Each is a tuple of `terminal * production size`.
local productions = {
    { 55, 1 }, { 54, 1 }, { 84, 1 }, { 84, 1 }, { 83, 3 }, { 82, 1 },
    { 82, 1 }, { 82, 1 }, { 82, 1 }, { 82, 1 }, { 82, 1 }, { 82, 1 },
    { 82, 1 }, { 82, 4 }, { 81, 2 }, { 81, 4 }, { 80, 3 }, { 80, 1 },
    { 80, 1 }, { 79, 1 }, { 79, 3 }, { 79, 3 }, { 79, 3 }, { 79, 3 },
    { 79, 3 }, { 79, 3 }, { 79, 3 }, { 79, 3 }, { 79, 3 }, { 79, 3 },
    { 79, 3 }, { 79, 3 }, { 79, 3 }, { 79, 3 }, { 78, 1 }, { 78, 3 },
    { 78, 2 }, { 78, 2 }, { 78, 2 }, { 77, 1 }, { 77, 3 }, { 77, 3 },
    { 76, 1 }, { 76, 1 }, { 75, 0 }, { 75, 2 }, { 75, 3 }, { 74, 1 },
    { 74, 3 }, { 74, 4 }, { 73, 2 }, { 72, 2 }, { 71, 0 }, { 71, 5 },
    { 70, 0 }, { 70, 2 }, { 69, 0 }, { 69, 1 }, { 68, 0 }, { 68, 1 },
    { 67, 1 }, { 67, 3 }, { 66, 1 }, { 66, 3 }, { 65, 1 }, { 65, 3 },
    { 64, 1 }, { 64, 3 }, { 63, 1 }, { 63, 3 }, { 63, 1 }, { 62, 3 },
    { 62, 3 }, { 62, 5 }, { 62, 4 }, { 62, 6 }, { 62, 8 }, { 62, 9 },
    { 62, 11 }, { 62, 7 }, { 62, 2 }, { 62, 4 }, { 62, 6 }, { 62, 5 },
    { 62, 1 }, { 62, 1 }, { 62, 1 }, { 62, 2 }, { 62, 3 }, { 61, 2 },
    { 60, 3 }, { 59, 0 }, { 59, 1 }, { 59, 3 }, { 58, 1 }, { 58, 3 },
    { 58, 5 }, { 57, 1 }, { 57, 1 }, { 56, 1 },
}

local f = false

--[[- The state machine used for our grammar.

Most LR(1) parsers will encode the transition table in a compact binary format,
optimised for space and fast lookups. However, without access to built-in
bitwise operations, this is harder to justify in Lua. Instead, the transition
table is a 2D lookup table of `action = transitions[state][value]`, where
`action` can be one of the following:

 - `action = false`: This transition is undefined, and thus a parse error. We
   use this (rather than nil) to ensure our tables are dense, and thus stored as
   arrays rather than maps.

 - `action > 0`: Shift this terminal or non-terminal onto the stack, then
   transition to `state = action`.

 - `action < 0`: Apply production `productions[-action]`. This production is a
   tuple composed of the next state and the number of values to pop from the
   stack.
]]
local transitions = {
    { -55, f, f, f, f, f, -55, -55, -55, f, f, f, -55, f, f, f, f, f, f, f, f, -55, f, f, f, -55, -55, f, -55, f, -55, -55, f, f, f, -55, f, f, f, -55, f, f, -55, f, f, f, f, f, f, f, -55, f, f, f, f, f, f, f, f, f, 2, f, f, f, f, f, f, f, f, 4, f, f, 195 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 3 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -51 },
    { 5, -45, f, f, f, f, 113, 114, 117, f, f, f, 9, f, f, f, f, f, f, f, f, 121, f, f, f, 133, 16, f, 146, f, 148, 158, f, f, f, -45, -45, -45, -45, 178, f, f, 181, f, f, f, f, f, f, f, 184, f, f, f, f, 34, f, f, f, f, f, 185, 186, f, 187, f, f, f, f, f, f, f, f, 192, 193, 24, f, f, f, f, 194 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 109, f, 43, 44 },
    { -9, -9, f, -9, -9, f, -9, -9, -9, -9, f, -9, -9, f, f, f, f, -9, -9, -9, -9, -9, f, -9, f, -9, -9, -9, -9, -9, -9, -9, f, f, -9, -9, -9, -9, -9, -9, f, f, -9, -9, -9, -9, -9, -9, f, -9, -9, -9, -9 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 108, f, f, 43, 44 },
    { -13, -13, f, -13, -13, f, -13, -13, -13, -13, f, -13, -13, f, f, f, f, -13, -13, -13, -13, -13, f, -13, f, -13, -13, -13, -13, -13, -13, -13, f, f, -13, -13, -13, -13, -13, -13, f, f, -13, -13, -13, -13, -13, -13, f, -13, -13, -13, -13 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 106, f, 43, 44 },
    { f, f, 6, f, 7, 8, f, f, f, f, 11, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, -92, f, f, f, f, f, 34, f, 95, 104, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 100, f, 40, 103, f, 43, 44 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 91, f, 43, 44 },
    { -12, -12, f, -12, -12, f, -12, -12, -12, -12, f, -12, -12, f, f, f, f, -12, -12, -12, -12, -12, f, -12, f, -12, -12, -12, -12, -12, -12, -12, f, f, -12, -12, -12, -12, -12, -12, f, f, -12, -12, -12, -12, -12, -12, f, -12, -12, -12, -12 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 90, f, f, 43, 44 },
    { -8, -8, f, -8, -8, f, -8, -8, -8, -8, f, -8, -8, f, f, f, f, -8, -8, -8, -8, -8, f, -8, f, -8, -8, -8, -8, -8, -8, -8, f, f, -8, -8, -8, -8, -8, -8, f, f, -8, -8, -8, -8, -8, -8, f, -8, -8, -8, -8 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 89, f, f, 43, 44 },
    { -43, -43, f, -43, -43, -43, -43, -43, -43, -43, -43, -43, -43, -43, f, f, f, -43, -43, -43, -43, -43, f, -43, -43, -43, -43, -43, -43, -43, -43, -43, f, -43, -43, -43, -43, -43, -43, -43, f, -43, -43, -43, -43, -43, -43, -43, -43, -43, -43, -43, -43 },
    { -44, -44, f, -44, -44, -44, -44, -44, -44, -44, -44, -44, -44, -44, f, f, f, -44, -44, -44, -44, -44, f, -44, -44, -44, -44, -44, -44, -44, -44, -44, f, -44, -44, -44, -44, -44, -44, -44, f, -44, -44, -44, -44, -44, -44, -44, -44, -44, -44, -44, -44 },
    { f, f, f, f, f, f, f, f, f, f, f, f, 19, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 29 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, 17, f, f, f, f, f, f, f, f, f, f, f, 20, f, f, f, f, -57, f, f, f, f, f, f, f, f, f, 21, f, f, f, f, f, f, f, f, f, f, 22, f, 26, f, f, f, f, f, f, 24, f, f, f, f, f, f, f, 28 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -4, f, -4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -3, f, -3 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -58, f, 23 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, 17, f, f, f, f, f, f, f, f, f, f, f, 20, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 21, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 24, f, f, f, f, f, f, f, 25 },
    { -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -62, f, -62 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 27 },
    { -5, f, f, f, f, f, -5, -5, -5, f, f, f, -5, f, f, f, f, f, f, f, f, -5, f, f, f, -5, -5, f, -5, f, -5, -5, f, f, f, f, -5, f, f, -5, f, f, -5, f, f, f, f, f, f, f, -5 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -61, f, -61 },
    { -55, f, f, f, f, f, -55, -55, -55, f, f, f, -55, f, f, f, f, f, f, f, f, -55, f, f, f, -55, -55, f, -55, f, -55, -55, f, f, f, f, -55, f, f, -55, f, f, -55, f, f, f, f, f, f, f, -55, f, f, f, f, f, f, f, f, f, 30, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 31 },
    { -14, -14, f, -14, -14, f, -14, -14, -14, -14, f, -14, -14, f, f, f, f, -14, -14, -14, -14, -14, f, -14, f, -14, -14, -14, -14, -14, -14, -14, f, f, -14, -14, -14, -14, -14, -14, f, f, -14, -14, -14, -14, -14, -14, f, -14, -14, -14, -14 },
    { -10, -10, f, -10, -10, f, -10, -10, -10, -10, f, -10, -10, f, f, f, f, -10, -10, -10, -10, -10, f, -10, f, -10, -10, -10, -10, -10, -10, -10, f, f, -10, -10, -10, -10, -10, -10, f, f, -10, -10, -10, -10, -10, -10, f, -10, -10, -10, -10 },
    { -11, -11, f, -11, -11, f, -11, -11, -11, -11, f, -11, -11, f, f, f, f, -11, -11, -11, -11, -11, f, -11, f, -11, -11, -11, -11, -11, -11, -11, f, f, -11, -11, -11, -11, -11, -11, f, f, -11, -11, -11, -11, -11, -11, f, -11, -11, -11, -11 },
    { -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48 },
    { -7, -7, f, -7, -7, f, -7, -7, -7, -7, f, -7, -7, f, f, f, f, -7, -7, -7, -7, -7, f, -7, f, -7, -7, -7, -7, -7, -7, -7, f, f, -7, -7, -7, -7, -7, -7, f, f, -7, -7, -7, -7, -7, -7, f, -7, -7, -7, -7 },
    { -6, -6, f, -6, -6, 37, -6, -6, -6, -6, 38, -6, 75, 10, f, f, f, -6, -6, -6, -6, -6, f, -6, f, -6, -6, -6, -6, -6, -6, -6, f, f, -6, -6, -6, -6, -6, -6, f, 82, -6, -6, -6, -6, -6, -6, 84, -6, -6, -6, -6, f, f, f, f, f, f, 86, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 88 },
    { -18, -18, f, -18, -18, -18, -18, -18, -18, -18, -18, -18, -18, -18, f, f, f, -18, -18, -18, -18, -18, f, -18, f, -18, -18, -18, -18, -18, -18, -18, f, f, -18, -18, -18, -18, -18, -18, f, -18, -18, -18, -18, -18, -18, -18, -18, -18, -18, -18, -18 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 45, f, 43, 44 },
    { -69, -69, f, -69, -69, -69, -69, -69, -69, -69, -69, -69, -69, -69, f, f, f, -69, -69, -69, -69, -69, f, -69, f, -69, -69, -69, -69, -69, -69, -69, f, f, -69, -69, -69, -69, -69, -69, f, -69, -69, -69, -69, -69, -69, -69, -69, -69, -69, -69, -69 },
    { -20, -20, f, -20, -20, f, -20, -20, -20, 41, f, -20, -20, f, f, f, f, -20, -20, -20, -20, -20, f, -20, f, -20, -20, -20, -20, -20, -20, -20, f, f, -20, -20, -20, -20, -20, -20, f, f, -20, -20, -20, -20, -20, -20, f, -20, -20, -20, -20 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 42, f, f, 43, 44 },
    { -36, -36, -36, -36, -36, -36, -36, -36, -36, 41, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36 },
    { -71, -71, f, -71, -71, -71, -71, -71, -71, -71, -71, -71, -71, -71, f, f, f, -71, -71, -71, -71, -71, f, -71, f, -71, -71, -71, -71, -71, -71, -71, f, f, -71, -71, -71, -71, -71, -71, f, -71, -71, -71, -71, -71, -71, -71, -71, -71, -71, -71, -71 },
    { -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35 },
    { f, f, f, f, 46, f, f, f, f, f, f, 54, f, f, f, f, f, 56, 48, 50, 62, f, f, 64, f, f, f, 66, f, 68, f, f, f, f, 70, f, f, f, f, f, f, f, f, 52, 74, f, 58, f, f, f, f, 72, 60 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 47, f, 43, 44 },
    { -24, -24, f, -24, -24, f, -24, -24, -24, f, f, -24, -24, f, f, f, f, -24, 48, 50, -24, -24, f, -24, f, -24, -24, -24, -24, -24, -24, -24, f, f, -24, -24, -24, -24, -24, -24, f, f, -24, 52, -24, -24, -24, -24, f, -24, -24, -24, -24 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 49, f, 43, 44 },
    { -25, -25, f, -25, -25, f, -25, -25, -25, f, f, -25, -25, f, f, f, f, -25, -25, -25, -25, -25, f, -25, f, -25, -25, -25, -25, -25, -25, -25, f, f, -25, -25, -25, -25, -25, -25, f, f, -25, -25, -25, -25, -25, -25, f, -25, -25, -25, -25 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 51, f, 43, 44 },
    { -27, -27, f, -27, -27, f, -27, -27, -27, f, f, -27, -27, f, f, f, f, -27, -27, -27, -27, -27, f, -27, f, -27, -27, -27, -27, -27, -27, -27, f, f, -27, -27, -27, -27, -27, -27, f, f, -27, -27, -27, -27, -27, -27, f, -27, -27, -27, -27 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 53, f, 43, 44 },
    { -26, -26, f, -26, -26, f, -26, -26, -26, f, f, -26, -26, f, f, f, f, -26, -26, -26, -26, -26, f, -26, f, -26, -26, -26, -26, -26, -26, -26, f, f, -26, -26, -26, -26, -26, -26, f, f, -26, -26, -26, -26, -26, -26, f, -26, -26, -26, -26 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 55, f, 43, 44 },
    { -22, -22, f, -22, 46, f, -22, -22, -22, f, f, -22, -22, f, f, f, f, 56, 48, 50, 62, -22, f, 64, f, -22, -22, 66, -22, 68, -22, -22, f, f, 70, -22, -22, -22, -22, -22, f, f, -22, 52, -22, -22, 58, -22, f, -22, -22, 72, 60 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 57, f, 43, 44 },
    { -30, -30, f, -30, 46, f, -30, -30, -30, f, f, -30, -30, f, f, f, f, -30, 48, 50, -30, -30, f, -30, f, -30, -30, -30, -30, -30, -30, -30, f, f, -30, -30, -30, -30, -30, -30, f, f, -30, 52, -30, -30, 58, -30, f, -30, -30, -30, 60 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 59, f, 43, 44 },
    { -28, -28, f, -28, 46, f, -28, -28, -28, f, f, -28, -28, f, f, f, f, -28, 48, 50, -28, -28, f, -28, f, -28, -28, -28, -28, -28, -28, -28, f, f, -28, -28, -28, -28, -28, -28, f, f, -28, 52, -28, -28, 58, -28, f, -28, -28, -28, 60 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 61, f, 43, 44 },
    { -23, -23, f, -23, -23, f, -23, -23, -23, f, f, -23, -23, f, f, f, f, -23, 48, 50, -23, -23, f, -23, f, -23, -23, -23, -23, -23, -23, -23, f, f, -23, -23, -23, -23, -23, -23, f, f, -23, 52, -23, -23, -23, -23, f, -23, -23, -23, -23 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 63, f, 43, 44 },
    { -31, -31, f, -31, 46, f, -31, -31, -31, f, f, -31, -31, f, f, f, f, -31, 48, 50, -31, -31, f, -31, f, -31, -31, -31, -31, -31, -31, -31, f, f, -31, -31, -31, -31, -31, -31, f, f, -31, 52, -31, -31, 58, -31, f, -31, -31, -31, 60 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 65, f, 43, 44 },
    { -32, -32, f, -32, 46, f, -32, -32, -32, f, f, -32, -32, f, f, f, f, -32, 48, 50, -32, -32, f, -32, f, -32, -32, -32, -32, -32, -32, -32, f, f, -32, -32, -32, -32, -32, -32, f, f, -32, 52, -32, -32, 58, -32, f, -32, -32, -32, 60 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 67, f, 43, 44 },
    { -33, -33, f, -33, 46, f, -33, -33, -33, f, f, -33, -33, f, f, f, f, -33, 48, 50, -33, -33, f, -33, f, -33, -33, -33, -33, -33, -33, -33, f, f, -33, -33, -33, -33, -33, -33, f, f, -33, 52, -33, -33, 58, -33, f, -33, -33, -33, 60 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 69, f, 43, 44 },
    { -34, -34, f, -34, 46, f, -34, -34, -34, f, f, -34, -34, f, f, f, f, -34, 48, 50, -34, -34, f, -34, f, -34, -34, -34, -34, -34, -34, -34, f, f, -34, -34, -34, -34, -34, -34, f, f, -34, 52, -34, -34, 58, -34, f, -34, -34, -34, 60 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 71, f, 43, 44 },
    { -29, -29, f, -29, 46, f, -29, -29, -29, f, f, -29, -29, f, f, f, f, -29, 48, 50, -29, -29, f, -29, f, -29, -29, -29, -29, -29, -29, -29, f, f, -29, -29, -29, -29, -29, -29, f, f, -29, 52, -29, -29, 58, -29, f, -29, -29, -29, 60 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 73, f, 43, 44 },
    { -21, -21, f, -21, 46, f, -21, -21, -21, f, f, -21, -21, f, f, f, f, 56, 48, 50, 62, -21, f, 64, f, -21, -21, 66, -21, 68, -21, -21, f, f, 70, -21, -21, -21, -21, -21, f, f, -21, 52, -21, -21, 58, -21, f, -21, -21, -21, 60 },
    { -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, -59, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, 76, f, 79, f, f, f, f, f, 39, f, 24, f, 40, 81, f, 43, 44 },
    { f, -60, f, f, f, f, -60, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -60, -60, -60, -60, f, f, f, f, f, f, -60, f, 77 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 78, f, 43, 44 },
    { -64, -64, f, f, 46, f, -64, -64, -64, f, f, 54, -64, f, f, f, f, 56, 48, 50, 62, -64, f, 64, f, -64, -64, 66, -64, 68, -64, -64, f, f, 70, -64, -64, -64, -64, -64, f, f, -64, 52, f, -64, 58, -64, f, f, -64, 72, 60 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 80 },
    { -17, -17, f, -17, -17, -17, -17, -17, -17, -17, -17, -17, -17, -17, f, f, f, -17, -17, -17, -17, -17, f, -17, f, -17, -17, -17, -17, -17, -17, -17, f, f, -17, -17, -17, -17, -17, -17, f, -17, -17, -17, -17, -17, -17, -17, -17, -17, -17, -17, -17 },
    { -63, -63, f, f, 46, f, -63, -63, -63, f, f, 54, -63, f, f, f, f, 56, 48, 50, 62, -63, f, 64, f, -63, -63, 66, -63, 68, -63, -63, f, f, 70, -63, -63, -63, -63, -63, f, f, -63, 52, f, -63, 58, -63, f, f, -63, 72, 60 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, 17, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 83 },
    { -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, 17, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 85 },
    { f, f, f, f, f, 37, f, f, f, f, f, f, 75, 10, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 86, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 87 },
    { -19, -19, f, -19, -19, -19, -19, -19, -19, -19, -19, -19, -19, -19, f, f, f, -19, -19, -19, -19, -19, f, -19, f, -19, -19, -19, -19, -19, -19, -19, f, f, -19, -19, -19, -19, -19, -19, f, -19, -19, -19, -19, -19, -19, -19, -19, -19, -19, -19, -19 },
    { -16, -16, f, -16, -16, -16, -16, -16, -16, -16, -16, -16, -16, -16, f, f, f, -16, -16, -16, -16, -16, f, -16, f, -16, -16, -16, -16, -16, -16, -16, f, f, -16, -16, -16, -16, -16, -16, f, -16, -16, -16, -16, -16, -16, -16, -16, -16, -16, -16, -16 },
    { -15, -15, f, -15, -15, -15, -15, -15, -15, -15, -15, -15, -15, -15, f, f, f, -15, -15, -15, -15, -15, f, -15, f, -15, -15, -15, -15, -15, -15, -15, f, f, -15, -15, -15, -15, -15, -15, f, -15, -15, -15, -15, -15, -15, -15, -15, -15, -15, -15, -15 },
    { -38, -38, -38, -38, -38, -38, -38, -38, -38, 41, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38 },
    { -39, -39, -39, -39, -39, -39, -39, -39, -39, 41, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39 },
    { f, f, f, f, 46, f, f, f, f, f, f, 54, f, f, f, f, f, 56, 48, 50, 62, f, f, 64, f, f, f, 66, f, 68, f, f, f, f, 70, f, f, f, f, f, f, f, f, 52, 92, f, 58, f, f, f, f, 72, 60 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 93 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 94, f, 43, 44 },
    { f, f, f, f, 46, f, -97, f, f, f, f, 54, f, f, f, f, f, 56, 48, 50, 62, f, f, 64, f, f, f, 66, f, 68, f, f, f, f, 70, f, f, f, f, f, f, f, f, 52, f, f, 58, -97, f, -97, f, 72, 60 },
    { f, f, f, f, f, f, 96, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 97, f, -93, f, f, f, f, f, f, 98 },
    { f, f, -98, f, -98, -98, f, f, f, f, -98, f, -98, -98, -98, -98, -98, f, f, f, f, f, -98, f, f, f, -98, f, -98, f, -98, f, -98, f, f, f, f, f, f, f, -98, f, f, f, f, f, f, f, f, -98 },
    { f, f, -99, f, -99, -99, f, f, f, f, -99, f, -99, -99, -99, -99, -99, f, f, f, f, f, -99, f, f, f, -99, f, -99, f, -99, f, -99, f, f, f, f, f, f, f, -99, f, f, f, f, f, f, f, f, -99 },
    { f, f, 6, f, 7, 8, f, f, f, f, 11, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, -92, f, f, f, f, f, 34, f, 95, 99, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 100, f, 40, 103, f, 43, 44 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -94 },
    { -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, 101, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100, -100 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 102, f, 43, 44 },
    { f, f, f, f, 46, f, -96, f, f, f, f, 54, f, f, f, f, f, 56, 48, 50, 62, f, f, 64, f, f, f, 66, f, 68, f, f, f, f, 70, f, f, f, f, f, f, f, f, 52, f, f, 58, -96, f, -96, f, 72, 60 },
    { f, f, f, f, 46, f, -95, f, f, f, f, 54, f, f, f, f, f, 56, 48, 50, 62, f, f, 64, f, f, f, 66, f, 68, f, f, f, f, 70, f, f, f, f, f, f, f, f, 52, f, f, 58, -95, f, -95, f, 72, 60 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 105 },
    { -91, -91, f, -91, -91, -91, -91, -91, -91, -91, -91, -91, -91, -91, f, f, f, -91, -91, -91, -91, -91, f, -91, f, -91, -91, -91, -91, -91, -91, -91, f, f, -91, -91, -91, -91, -91, -91, f, -91, -91, -91, -91, -91, -91, -91, -91, -91, -91, -91, -91 },
    { f, f, f, f, 46, f, f, f, f, f, f, 54, f, f, f, f, f, 56, 48, 50, 62, f, f, 64, f, f, f, 66, f, 68, f, f, f, f, 70, f, f, f, f, f, f, f, f, 52, f, 107, 58, f, f, f, f, 72, 60 },
    { -70, -70, f, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70, f, f, f, -70, -70, -70, -70, -70, f, -70, f, -70, -70, -70, -70, -70, -70, -70, f, f, -70, -70, -70, -70, -70, -70, f, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70 },
    { -37, -37, -37, -37, -37, -37, -37, -37, -37, 41, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37 },
    { f, f, f, f, 46, f, f, f, f, f, f, 54, f, f, f, f, f, 56, 48, 50, 62, f, f, 64, f, f, f, 66, f, 68, f, f, f, f, 70, f, f, f, f, f, f, f, 110, 52, f, f, 58, f, f, f, f, 72, 60 },
    { -55, f, f, f, f, f, -55, -55, -55, f, f, f, -55, f, f, f, f, f, f, f, f, -55, f, f, f, -55, -55, f, -55, f, -55, -55, f, f, f, f, -55, f, f, -55, f, f, -55, f, f, f, f, f, f, f, -55, f, f, f, f, f, f, f, f, f, 111, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 112 },
    { -74, -74, f, f, f, f, -74, -74, -74, f, f, f, -74, f, f, f, f, f, f, f, f, -74, f, f, f, -74, -74, f, -74, f, -74, -74, f, f, f, -74, -74, -74, -74, -74, f, f, -74, f, f, f, f, f, f, f, -74 },
    { -86, -86, f, f, f, f, -86, -86, -86, f, f, f, -86, f, f, f, f, f, f, f, f, -86, f, f, f, -86, -86, f, -86, f, -86, -86, f, f, f, -86, -86, -86, -86, -86, f, f, -86, f, f, f, f, f, f, f, -86 },
    { f, -59, 6, f, 7, 8, -59, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, -59, -59, -59, -59, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, 76, f, 115, f, f, f, f, f, 39, f, 24, f, 40, 81, f, 43, 44 },
    { f, -46, f, f, f, f, 116, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -46, -46, -46, -46 },
    { f, -47, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -47, -47, -47, -47 },
    { -55, -55, f, f, f, f, -55, -55, -55, f, f, f, -55, f, f, f, f, f, f, f, f, -55, f, f, f, -55, -55, f, -55, f, -55, -55, f, f, f, f, f, f, f, -55, f, f, -55, f, f, f, f, f, f, f, -55, f, f, f, f, f, f, f, f, f, 118, f, f, f, f, f, f, f, f, 4 },
    { f, 119 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 120, f, 43, 44 },
    { -75, -75, f, f, 46, f, -75, -75, -75, f, f, 54, -75, f, f, f, f, 56, 48, 50, 62, -75, f, 64, f, -75, -75, 66, -75, 68, -75, -75, f, f, 70, -75, -75, -75, -75, -75, f, f, -75, 52, f, f, 58, f, f, f, -75, 72, 60 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, 17, f, 122, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 127, f, f, f, f, f, f, f, 128, f, f, f, f, f, f, f, f, f, f, f, 24 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, 17, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 123, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 24 },
    { f, f, f, f, f, f, f, f, f, f, f, f, 19, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 124 },
    { -55, f, f, f, f, f, -55, -55, -55, f, f, f, -55, f, f, f, f, f, f, f, f, -55, f, f, f, -55, -55, f, -55, f, -55, -55, f, f, f, f, -55, f, f, -55, f, f, -55, f, f, f, f, f, f, f, -55, f, f, f, f, f, f, f, f, f, 125, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 126 },
    { -83, -83, f, f, f, f, -83, -83, -83, f, f, f, -83, f, f, f, f, f, f, f, f, -83, f, f, f, -83, -83, f, -83, f, -83, -83, f, f, f, -83, -83, -83, -83, -83, f, f, -83, f, f, f, f, f, f, f, -83 },
    { -67, -67, f, f, f, f, -67, -67, -67, f, f, f, -67, f, f, f, f, f, f, f, f, -67, f, f, f, -67, -67, f, -67, f, -67, -67, f, -67, f, -67, -67, -67, -67, -67, f, f, -67, f, f, f, f, -67, f, f, -67 },
    { -81, -81, f, f, f, f, -81, -81, -81, f, f, f, -81, f, f, f, f, f, f, f, f, -81, f, f, f, -81, -81, f, -81, f, -81, -81, f, 129, f, -81, -81, -81, -81, -81, f, f, -81, f, f, f, f, 131, f, f, -81 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, 130, f, f, f, f, f, f, f, 39, f, 24, f, 40, 81, f, 43, 44 },
    { -82, -82, f, f, f, f, -82, -82, -82, f, f, f, -82, f, f, f, f, f, f, f, f, -82, f, f, f, -82, -82, f, -82, f, -82, -82, f, f, f, -82, -82, -82, -82, -82, f, f, -82, f, f, f, f, 77, f, f, -82 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, 17, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 132, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 24 },
    { -68, -68, f, f, f, f, -68, -68, -68, f, f, f, -68, f, f, f, f, f, f, f, f, -68, f, f, -68, -68, -68, f, -68, f, -68, -68, f, -68, f, -68, -68, -68, -68, -68, f, f, -68, f, f, f, f, -68, f, f, -68 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 134, f, 43, 44 },
    { f, f, f, 135, 46, f, f, f, f, f, f, 54, f, f, f, f, f, 56, 48, 50, 62, f, f, 64, f, f, f, 66, f, 68, f, f, f, f, 70, f, f, f, f, f, f, f, f, 52, f, f, 58, f, f, f, f, 72, 60 },
    { -55, f, f, f, f, f, -55, -55, -55, f, f, f, -55, f, f, f, f, f, f, f, f, -55, f, f, f, -55, -55, f, -55, f, -55, -55, f, f, f, f, -55, -55, -55, -55, f, f, -55, f, f, f, f, f, f, f, -55, f, f, f, f, f, f, f, f, f, 136, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -53, -53, -53, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 137 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 138, 139, 143 },
    { -76, -76, f, f, f, f, -76, -76, -76, f, f, f, -76, f, f, f, f, f, f, f, f, -76, f, f, f, -76, -76, f, -76, f, -76, -76, f, f, f, -76, -76, -76, -76, -76, f, f, -76, f, f, f, f, f, f, f, -76 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 140, f, 43, 44 },
    { f, f, f, 141, 46, f, f, f, f, f, f, 54, f, f, f, f, f, 56, 48, 50, 62, f, f, 64, f, f, f, 66, f, 68, f, f, f, f, 70, f, f, f, f, f, f, f, f, 52, f, f, 58, f, f, f, f, 72, 60 },
    { -55, f, f, f, f, f, -55, -55, -55, f, f, f, -55, f, f, f, f, f, f, f, f, -55, f, f, f, -55, -55, f, -55, f, -55, -55, f, f, f, f, -55, -55, -55, -55, f, f, -55, f, f, f, f, f, f, f, -55, f, f, f, f, f, f, f, f, f, 142, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -54, -54, -54 },
    { -55, f, f, f, f, f, -55, -55, -55, f, f, f, -55, f, f, f, f, f, f, f, f, -55, f, f, f, -55, -55, f, -55, f, -55, -55, f, f, f, f, -55, f, f, -55, f, f, -55, f, f, f, f, f, f, f, -55, f, f, f, f, f, f, f, f, f, 144, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 145 },
    { -77, -77, f, f, f, f, -77, -77, -77, f, f, f, -77, f, f, f, f, f, f, f, f, -77, f, f, f, -77, -77, f, -77, f, -77, -77, f, f, f, -77, -77, -77, -77, -77, f, f, -77, f, f, f, f, f, f, f, -77 },
    { f, f, f, f, f, -44, f, f, f, f, -44, f, -44, -44, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, 17, f, f, f, f, -44, f, f, f, f, f, f, f, -44, f, f, f, f, f, -44, -44, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 147 },
    { -88, -88, f, f, f, f, -88, -88, -88, f, f, f, -88, f, f, f, f, f, f, f, f, -88, f, f, f, -88, -88, f, -88, f, -88, -88, f, f, f, -88, -88, -88, -88, -88, f, f, -88, f, f, f, f, f, f, f, -88 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, 17, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 149, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 24, 150 },
    { f, f, f, f, f, f, f, f, f, f, f, f, -40, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -40, f, f, f, f, f, f, -40 },
    { f, f, f, f, f, f, f, f, f, f, f, f, 19, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 151, f, f, f, f, f, f, 153, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 155 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 152 },
    { f, f, f, f, f, f, f, f, f, f, f, f, -41, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -41, f, f, f, f, f, f, -41 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 154 },
    { f, f, f, f, f, f, f, f, f, f, f, f, -42, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -42, f, f, f, f, f, f, -42 },
    { -55, f, f, f, f, f, -55, -55, -55, f, f, f, -55, f, f, f, f, f, f, f, f, -55, f, f, f, -55, -55, f, -55, f, -55, -55, f, f, f, f, -55, f, f, -55, f, f, -55, f, f, f, f, f, f, f, -55, f, f, f, f, f, f, f, f, f, 156, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 157 },
    { -84, -84, f, f, f, f, -84, -84, -84, f, f, f, -84, f, f, f, f, f, f, f, f, -84, f, f, f, -84, -84, f, -84, f, -84, -84, f, f, f, -84, -84, -84, -84, -84, f, f, -84, f, f, f, f, f, f, f, -84 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, 17, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 159, f, f, f, f, f, f, f, 172, f, f, f, f, f, f, f, f, f, f, f, 24 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -67, f, f, f, f, f, f, f, f, 160, f, f, f, f, f, f, f, f, f, f, f, f, f, -67 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 161, f, 43, 44 },
    { f, f, f, f, 46, f, f, f, f, f, f, 54, f, f, f, f, f, 56, 48, 50, 62, f, f, 64, f, f, f, 66, f, 68, f, f, f, f, 70, f, f, f, f, f, f, f, f, 52, f, f, 58, 162, f, f, f, 72, 60 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 163, f, 43, 44 },
    { f, f, f, f, 46, f, f, f, f, f, f, 54, f, f, f, f, f, 56, 48, 50, 62, f, f, 64, f, f, f, 66, f, 68, f, f, f, f, 70, f, f, f, f, f, f, f, 164, 52, f, f, 58, 167, f, f, f, 72, 60 },
    { -55, f, f, f, f, f, -55, -55, -55, f, f, f, -55, f, f, f, f, f, f, f, f, -55, f, f, f, -55, -55, f, -55, f, -55, -55, f, f, f, f, -55, f, f, -55, f, f, -55, f, f, f, f, f, f, f, -55, f, f, f, f, f, f, f, f, f, 165, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 166 },
    { -78, -78, f, f, f, f, -78, -78, -78, f, f, f, -78, f, f, f, f, f, f, f, f, -78, f, f, f, -78, -78, f, -78, f, -78, -78, f, f, f, -78, -78, -78, -78, -78, f, f, -78, f, f, f, f, f, f, f, -78 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, f, f, f, f, f, f, f, f, 39, f, 24, f, 40, 168, f, 43, 44 },
    { f, f, f, f, 46, f, f, f, f, f, f, 54, f, f, f, f, f, 56, 48, 50, 62, f, f, 64, f, f, f, 66, f, 68, f, f, f, f, 70, f, f, f, f, f, f, f, 169, 52, f, f, 58, f, f, f, f, 72, 60 },
    { -55, f, f, f, f, f, -55, -55, -55, f, f, f, -55, f, f, f, f, f, f, f, f, -55, f, f, f, -55, -55, f, -55, f, -55, -55, f, f, f, f, -55, f, f, -55, f, f, -55, f, f, f, f, f, f, f, -55, f, f, f, f, f, f, f, f, f, 170, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 171 },
    { -79, -79, f, f, f, f, -79, -79, -79, f, f, f, -79, f, f, f, f, f, f, f, f, -79, f, f, f, -79, -79, f, -79, f, -79, -79, f, f, f, -79, -79, -79, -79, -79, f, f, -79, f, f, f, f, f, f, f, -79 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 173, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 131 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, 174, f, f, f, f, f, f, f, 39, f, 24, f, 40, 81, f, 43, 44 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 175, f, f, f, f, 77 },
    { -55, f, f, f, f, f, -55, -55, -55, f, f, f, -55, f, f, f, f, f, f, f, f, -55, f, f, f, -55, -55, f, -55, f, -55, -55, f, f, f, f, -55, f, f, -55, f, f, -55, f, f, f, f, f, f, f, -55, f, f, f, f, f, f, f, f, f, 176, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 177 },
    { -80, -80, f, f, f, f, -80, -80, -80, f, f, f, -80, f, f, f, f, f, f, f, f, -80, f, f, f, -80, -80, f, -80, f, -80, -80, f, f, f, -80, -80, -80, -80, -80, f, f, -80, f, f, f, f, f, f, f, -80 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, 17, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 179 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 180 },
    { -89, -89, f, f, f, f, -89, -89, -89, f, f, f, -89, f, f, f, f, f, f, f, f, -89, f, f, f, -89, -89, f, -89, f, -89, -89, f, f, f, -89, -89, -89, -89, -89, f, f, -89, f, f, f, f, f, f, f, -89 },
    { -55, f, f, f, f, f, -55, -55, -55, f, f, f, -55, f, f, f, f, f, f, f, f, -55, f, f, f, -55, -55, f, -55, f, -55, -55, f, f, f, f, -55, f, f, -55, f, f, -55, f, f, f, f, f, f, f, -55, f, f, f, f, f, f, f, f, f, 182, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 183 },
    { -72, -72, f, f, f, f, -72, -72, -72, f, f, f, -72, f, f, f, f, f, f, f, f, -72, f, f, f, -72, -72, f, -72, f, -72, -72, f, f, f, -72, -72, -72, -72, -72, f, f, -72, f, f, f, f, f, f, f, -72 },
    { -87, -87, f, f, f, f, -87, -87, -87, f, f, f, -87, f, f, f, f, f, f, f, f, -87, f, f, f, -87, -87, f, -87, f, -87, -87, f, f, f, -87, -87, -87, -87, -87, f, f, -87, f, f, f, f, f, f, f, -87 },
    { -56, -56, f, f, f, f, -56, -56, -56, f, f, f, -56, f, f, f, f, f, f, f, f, -56, f, f, f, -56, -56, f, -56, f, -56, -56, f, f, f, -56, -56, -56, -56, -56, f, f, -56, f, f, f, f, f, f, f, -56 },
    { f, f, f, f, f, 37, f, f, f, f, 38, f, 75, 10, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 82, f, f, f, f, f, f, 84, f, f, f, f, f, f, f, f, f, f, 86, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 88 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 188, f, f, f, f, f, f, f, f, f, f, f, f, f, 190 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, 189, f, f, f, f, f, f, f, 39, f, 24, f, 40, 81, f, 43, 44 },
    { -73, -73, f, f, f, f, -73, -73, -73, f, f, f, -73, f, f, f, f, f, f, f, f, -73, f, f, f, -73, -73, f, -73, f, -73, -73, f, f, f, -73, -73, -73, -73, -73, f, f, -73, f, f, f, f, 77, f, f, -73 },
    { f, f, f, f, f, f, f, f, f, f, f, f, 9, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, 17, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, f, f, f, 186, f, f, f, f, f, f, f, f, f, f, 191, f, 24, f, f, f, f, 43 },
    { f, f, f, f, f, -69, f, f, f, f, -69, f, -69, -69, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -66, f, f, f, f, f, f, f, -69, f, f, f, f, f, -66, -69 },
    { f, f, f, f, f, -69, f, f, f, f, -69, f, -69, -69, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -65, f, f, f, f, f, f, f, -69, f, f, f, f, f, -65, -69 },
    { -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90, -90 },
    { -85, -85, f, f, f, -71, -85, -85, -85, f, -71, f, -71, -71, f, f, f, f, f, f, f, -85, f, f, f, -85, -85, f, -85, f, -85, -85, f, f, f, -85, -85, -85, -85, -85, f, -71, -85, f, f, f, f, f, -71, f, -85 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -1 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, 17, f, 18, f, 32, f, f, f, f, f, f, f, 33, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 34, f, f, f, 35, f, f, 36, f, f, 197, f, f, f, f, f, 199, f, 39, f, 24, f, 40, 81, f, 43, 44 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 198, f, f, f, f, f, f, f, f, f, f, f, 77 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -52 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -2 },
}

--- Run the parser across a sequence of tokens.
--
-- @tparam table context The current parser context.
-- @tparam function get_next A stateful function which returns the next token.
-- @treturn boolean Whether the parse succeeded or not.
local function parse(context, get_next, start)
    local stack, stack_n = { start or 1, 1, 1 }, 1
    local reduce_stack = {}

    while true do
        local token, token_start, token_end = get_next()
        local state = stack[stack_n]
        local action = transitions[state][token]

        if not action then -- Error
            return handle_error(context, stack, stack_n, token, token_start, token_end)
        elseif action >= 0 then -- Shift
            stack_n = stack_n + 3
            stack[stack_n], stack[stack_n + 1], stack[stack_n + 2] = action, token_start, token_end
        elseif action >= -2 then -- Accept
            return true
        else -- Reduce
            -- Reduction is quite complex to get right, as the error code expects the parser
            -- to be shifting rather than reducing. Menhir achieves this by making the parser
            -- stack be immutable, but that's hard to do efficiently in Lua: instead we track
            -- what symbols we've pushed/popped, and only perform this change when we're ready
            -- to shift again.

            local popped, pushed = 0, 0
            while true do
                -- Look at the current item to reduce
                local reduce = productions[-action]
                local terminal, to_pop = reduce[1], reduce[2]

                -- Find the state at the start of this production. If to_pop == 0
                -- then use the current state.
                local lookback = state
                if to_pop > 0 then
                    pushed = pushed - to_pop
                    if pushed <= 0 then
                        -- If to_pop >= pushed, then clear the reduction stack
                        -- and consult the normal stack.
                        popped = popped - pushed
                        pushed = 0
                        lookback = stack[stack_n - popped * 3]
                    else
                        -- Otherwise consult the stack of temporary reductions.
                        lookback = reduce_stack[pushed]
                    end
                end

                state = transitions[lookback][terminal]
                if not state or state <= 0 then error("reduce must shift!") end

                -- And fetch the next action
                action = transitions[state][token]

                if not action then -- Error
                    return handle_error(context, stack, stack_n, token, token_start, token_end)
                elseif action >= 0 then -- Shift
                    break
                elseif action >= -2 then -- Accept
                    return true
                else
                    pushed = pushed + 1
                    reduce_stack[pushed] = state
                end
            end

            if popped == 1 and pushed == 0 then
                -- Handle the easy case: Popped one item and replaced it with another
                stack[stack_n] = state
            else
                -- Otherwise pop and push.
                -- FIXME: The positions of everything here are entirely wrong.
                local end_pos = stack[stack_n + 2]
                stack_n = stack_n - popped * 3
                local start_pos = stack[stack_n + 1]

                for i = 1, pushed do
                    stack_n = stack_n + 3
                    stack[stack_n], stack[stack_n + 1], stack[stack_n + 2] = reduce_stack[i], end_pos, end_pos
                end

                stack_n = stack_n + 3
                stack[stack_n], stack[stack_n + 1], stack[stack_n + 2] = state, start_pos, end_pos
            end

            -- Shift the token onto the stack
            stack_n = stack_n + 3
            stack[stack_n], stack[stack_n + 1], stack[stack_n + 2] = action, token_start, token_end
        end
    end
end

return {
    tokens = tokens,
    parse = parse,
    repl_exprs = 196, --[[- The repl_exprs starting state. ]]
    program = 1, --[[- The program starting state. ]]
}
